//*****************************************************************************
//
// ektm4c129_qssi_example.c - Example demonstrating how to configure the QSSI
//			 module for accessing a Serial Flash in Adv-Bi-Quad Mode
//
// Copyright (c) 2014 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
//
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met:
//
//   Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the
//   distribution.
//
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision 2.1.0.12573 of the Tiva Firmware Development Package.
//
//*****************************************************************************
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_gpio.h"
#include "inc/hw_ssi.h"
#include "inc/hw_sysctl.h"
#include "inc/hw_uart.h"
#include "driverlib/debug.h"
#include "driverlib/fpu.h"
#include "driverlib/gpio.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/pin_map.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "utils/uartstdio.h"

//*****************************************************************************
//
// The number of Bytes to be Sent or received
//
//*****************************************************************************
#define NUM_SSI_DATA 				256

//*****************************************************************************
//
// The QSSI Flash Memory Access Instructions
//
//*****************************************************************************
#define INS_WRITE_ENABLE 			0x06
#define INS_WRITE_STATUS_REGISTER   0x01
#define INS_READ_STATUS_REGISTER1   0x05
#define INS_SECTOR_ERASE_4KB 		0x20
#define INS_ADV_PAGE_PROGRAM 		0x02
#define INS_QUAD_PAGE_PROGRAM		0x38
#define INS_ADV_READ_DATA  			0x03
#define INS_BI_READ_DATA  			0x3B
#define INS_QUAD_READ_DATA  		0x6B

//*****************************************************************************
//
// Dummy Byte when Padding or during Receive operations
//
//*****************************************************************************
#define DUMMY_BYTE       			0x00

//*****************************************************************************
//
// The QSSI Flash memory Status Register Bits for Polling Operations
//
//*****************************************************************************
#define FLASH_WIP_BIT				0x01
#define FLASH_WEL_BIT				0x02
#define FLASH_QE_BIT				0x40

//*****************************************************************************
//
// The QSSI Flash memory Configuration Register Bits Default State
//
//*****************************************************************************
#define FLASH_CFG_REG				0x07

//*****************************************************************************
//
// Global Variables for Transmit and Receive Buffers
//
//*****************************************************************************
uint32_t pui32DataTx[NUM_SSI_DATA];
uint32_t pui32DataRx[NUM_SSI_DATA];

//*****************************************************************************
//
// This function sets up SSI External Flash ID Read
//
//*****************************************************************************
uint32_t SSILibSendReadIDAdvMode(uint32_t ui32Base)
{
   uint32_t ui32Idx, ui32Receive;
   uint32_t ui32MfgID;
   uint32_t ui32DevID;
   uint32_t ui32ReadID[]={0xaa,0xaa};
   uint32_t ui32ReadIDFlash;
   uint8_t  ui8InstrReadID[] = {0x90,0x00,0x00,0x00};

   for(ui32Idx = 0; ui32Idx < sizeof(ui8InstrReadID) / sizeof(ui8InstrReadID[0]); ui32Idx++)
   {
     SSIDataPut(ui32Base, ui8InstrReadID[ui32Idx]);
   }

   SSIAdvModeSet(ui32Base,SSI_ADV_MODE_READ_WRITE);
   SSIDataPut(ui32Base, 0x00);
   SSIDataGet(ui32Base, &ui32Receive);
   ui32ReadID[0] = ui32Receive;
   SSIAdvDataPutFrameEnd(ui32Base,0x00);
   SSIDataGet(ui32Base, &ui32Receive);
   ui32ReadID[1] = ui32Receive;
   ui32MfgID = ui32ReadID[0];
   ui32DevID = ui32ReadID[1];
   ui32ReadIDFlash = ui32MfgID;

   ui32ReadIDFlash = ui32ReadIDFlash << 8;
   ui32ReadIDFlash |= ui32DevID;

   SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);

   return ui32ReadIDFlash;
}

//*****************************************************************************
//
// This function sets up SSI External Flash Read of Status Register
//
//*****************************************************************************
uint8_t
SSILibSendReadStatusRegister(uint32_t ui32Base, uint8_t ui8FlashCommand)
{
    uint32_t ui32Status = 0;
    uint8_t ui8FlashStatus;

    SSIAdvFrameHoldEnable(ui32Base);
    SSIDataPut(ui32Base,ui8FlashCommand);
    SSIAdvModeSet(ui32Base,SSI_ADV_MODE_READ_WRITE);
    SSIAdvDataPutFrameEnd(ui32Base,DUMMY_BYTE);
    while(SSIBusy(ui32Base));
    SSIDataGet(ui32Base, &ui32Status);
    ui8FlashStatus = (uint8_t)ui32Status;
    SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);

    return ui8FlashStatus;
}


//*****************************************************************************
//
// This function sets up SSI External Flash Polling of Busy Bit Check
//
//*****************************************************************************
void
SSILibDeviceBusyCheck(uint32_t ui32Base)
{
   uint8_t ui8TempData=0xFF;

   ui8TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);

   while((ui8TempData & FLASH_WIP_BIT) == FLASH_WIP_BIT)
   {
      ui8TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
   }
}

//*****************************************************************************
//
// This function sets up SSI External Flash Program in Adv/Quad Mode
//
//*****************************************************************************
bool
SSILibSendPageProgram(uint32_t ui32Base, uint32_t ui32Address, uint8_t ui8FlashCommand)
{
    uint32_t ui32TempAddr, ui32TempData;
    uint8_t ui8AddrByte1, ui8AddrByte2, ui8AddrByte3;

    ui32TempAddr = ui32Address >> 16 ;
    ui8AddrByte1 = (uint8_t)ui32TempAddr;
    ui32TempAddr = ui32Address >> 8 ;
    ui8AddrByte2 = (uint8_t)ui32TempAddr;
    ui8AddrByte3 = (uint8_t)ui32Address;

    //
    // Check if the Command to Program the Flash is either Advanced or Quad Mode
    // If not then return false. if yes then first set the WEL bit
    //
    if((ui8FlashCommand == INS_ADV_PAGE_PROGRAM) || (ui8FlashCommand == INS_QUAD_PAGE_PROGRAM))
    {
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
    }
    else
    {
    	return (0);
    }

    SSIAdvFrameHoldEnable(ui32Base);
    SSIAdvDataPutFrameEnd(ui32Base,INS_WRITE_ENABLE);
    while(SSIBusy(ui32Base));

    ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
    while((ui32TempData & FLASH_WIP_BIT) == FLASH_WIP_BIT) {
      ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
    }

    //
    // Now Set the Quad Enable Bit if Quad Mode Program is requested
    //
    if(ui8FlashCommand == INS_QUAD_PAGE_PROGRAM)
    {
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
        SSIAdvFrameHoldEnable(ui32Base);
        SSIDataPut(ui32Base,INS_WRITE_STATUS_REGISTER);
        SSIDataPut(ui32Base,(FLASH_QE_BIT));
        SSIAdvDataPutFrameEnd(ui32Base,FLASH_CFG_REG);
        while(SSIBusy(ui32Base));

        //
        // Check if Quad Mode setting has been completed
        //
        ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
        while((ui32TempData & FLASH_WIP_BIT) == FLASH_WIP_BIT) {
          ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
        }

        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
        SSIAdvFrameHoldEnable(ui32Base);
        SSIAdvDataPutFrameEnd(ui32Base,INS_WRITE_ENABLE);
        while(SSIBusy(ui32Base));

        //
        // Set the WIP bit for Quad Mode Programming and check if
        // it has been completed as well.
        //
        ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
        while((ui32TempData & FLASH_WIP_BIT) == FLASH_WIP_BIT) {
          ui32TempData = SSILibSendReadStatusRegister(ui32Base,INS_READ_STATUS_REGISTER1);
        }
    }

    if(ui8FlashCommand == INS_ADV_PAGE_PROGRAM)
    {
    	SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
        SSIAdvFrameHoldEnable(ui32Base);
        SSIDataPut(ui32Base,ui8FlashCommand);
    }
    if(ui8FlashCommand == INS_QUAD_PAGE_PROGRAM)
    {
    	SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
        SSIAdvFrameHoldEnable(ui32Base);
        SSIDataPut(ui32Base,ui8FlashCommand);
    	SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_QUAD_WRITE);
    }
    SSIDataPut(ui32Base,ui8AddrByte1);
    SSIDataPut(ui32Base,ui8AddrByte2);
    SSIDataPut(ui32Base,ui8AddrByte3);

    //
    // Return Success
    //
    return (1);
}

//*****************************************************************************
//
// This function sets up SSI External Flash Read of Data in Advanced/Bi/Quad
// mode
//
//*****************************************************************************
bool
SSILibSendReadDataAdvBiQuad(uint32_t ui32Base, uint32_t ui32Address, uint8_t ui8FlashCommand)
{
    uint32_t ui32TempAddr;
    uint8_t ui8AddrByte1, ui8AddrByte2, ui8AddrByte3;

    //
    // Pre Format the Address
    //
    ui32TempAddr = ui32Address >> 16 ;
    ui8AddrByte1 = (uint8_t)ui32TempAddr;
    ui32TempAddr = ui32Address >> 8 ;
    ui8AddrByte2 = (uint8_t)ui32TempAddr;
    ui8AddrByte3 = (uint8_t)ui32Address;

    //
    // Check if Previous operation of Program or Erase has completed
    //
    SSILibDeviceBusyCheck(ui32Base);

    //
    // Check if the Command to Program the Flash is either Advanced or Quad Mode
    // If not then return false. if yes then first set the WEL bit
    //
    if((ui8FlashCommand == INS_ADV_READ_DATA) || (ui8FlashCommand == INS_BI_READ_DATA)
    		|| (ui8FlashCommand == INS_QUAD_READ_DATA))
    {
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
    }
    else
    {
    	return (0);
    }

    SSIAdvFrameHoldEnable(ui32Base);
    SSIDataPut(ui32Base,ui8FlashCommand);
    SSIDataPut(ui32Base,ui8AddrByte1);
    SSIDataPut(ui32Base,ui8AddrByte2);
    SSIDataPut(ui32Base,ui8AddrByte3);

    if(ui8FlashCommand == INS_ADV_READ_DATA)
    {
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_READ_WRITE);
    }
    if(ui8FlashCommand == INS_BI_READ_DATA)
    {
        SSIDataPut(ui32Base,DUMMY_BYTE);
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_BI_READ);
    }
    if(ui8FlashCommand == INS_QUAD_READ_DATA)
    {
        SSIDataPut(ui32Base,DUMMY_BYTE);
        SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_QUAD_READ);
    }

	return (1);
}

//*****************************************************************************
//
// This function sets up SSI External Flash Erase
//
//*****************************************************************************
void
SSILibSendEraseCommand (uint32_t ui32Base, uint32_t ui32Address, uint8_t ui8FlashCommand)
{
    uint32_t ui32TempAddr;
    uint8_t ui8AddrByte1, ui8AddrByte2, ui8AddrByte3;

    ui32TempAddr = ui32Address >> 16 ;

    ui8AddrByte1 = (uint8_t)ui32TempAddr;
    ui32TempAddr = ui32Address >> 8 ;
    ui8AddrByte2 = (uint8_t)ui32TempAddr;
    ui8AddrByte3 = (uint8_t)ui32Address;

    SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
    SSIAdvFrameHoldEnable(ui32Base);
    SSIAdvDataPutFrameEnd(ui32Base,INS_WRITE_ENABLE);
    while(SSIBusy(ui32Base));

    //
    // Check if Previous operation of Program or Erase has completed
    //
    SSILibDeviceBusyCheck(ui32Base);

    SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
    SSIAdvFrameHoldEnable(ui32Base);
    SSIDataPut(ui32Base,ui8FlashCommand);
    SSIDataPut(ui32Base,ui8AddrByte1);
    SSIDataPut(ui32Base,ui8AddrByte2);
    SSIAdvDataPutFrameEnd(ui32Base,ui8AddrByte3);
    while(SSIBusy(ui32Base));

    //
    // Wait till the erase is completed
    //
    SSILibDeviceBusyCheck(ui32Base);

}

//*****************************************************************************
//
// This function sets up UART0 to be used for a console to display information
// as the example is running.
//
//*****************************************************************************
void
InitConsole(void)
{
    //
    // Enable GPIO port A which is used for UART0 pins.
    // TODO: change this to whichever GPIO port you are using.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Configure the pin muxing for UART0 functions on port A0 and A1.
    // This step is not necessary if your part does not support pin muxing.
    // TODO: change this to select the port/pin you are using.
    //
    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);

    //
    // Enable UART0 so that we can configure the clock.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Use the internal 16MHz oscillator as the UART clock source.
    //
    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);

    //
    // Select the alternate (UART) function for these pins.
    // TODO: change this to select the port/pin you are using.
    //
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UART for console I/O.
    //
    UARTStdioConfig(0, 115200, 16000000);
}

//*****************************************************************************
//
// Configure SSI1 in master Freescale (SPI) mode.  This example will send out
// 256 bytes in Advanced, Bi and Quad Mode and then read the data in Advanced,
// Bi and Quad Mode. Once the data check is completed it shall Erase the Serial
// Flash to return the device to it's original state.
//
//*****************************************************************************
int
main(void)
{
    uint32_t pui32Dummy[1];
    uint32_t ui32Index;
    uint32_t ui32DeviceID;
    uint32_t ui32SysClockFreq;
    bool     bErrorFlag;

    //
    // Set the error flag to false here. if it gets set at any point the
    // program shall exit without any further execution
    //
   	bErrorFlag = false;

   	//
    // Set the clocking to run directly from the external crystal/oscillator.
    // TODO: The SYSCTL_XTAL_ value must be changed to match the value of the
    // crystal on your board.
    //
    ui32SysClockFreq = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                           SYSCTL_OSC_MAIN |
                                           SYSCTL_USE_PLL |
                                           SYSCTL_CFG_VCO_480), 120000000);
    //
    // Set up the serial console to use for displaying messages.  This is
    // just for this example program and is not needed for SSI operation.
    //
    InitConsole();

    //
    // Display the setup on the console.
    //
    UARTprintf("\033[2J\033[H");
    UARTprintf("\rSSI1 To EK-TM4C129 FLASH TRANSFER\n\n");

    //
    // The SSI1 peripheral must be disabled, reset and re enabled for use.
    // Wait till the Peripheral ready is not asserted
    //
    SysCtlPeripheralDisable(SYSCTL_PERIPH_SSI1);
    SysCtlPeripheralReset(SYSCTL_PERIPH_SSI1);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI1);
    while(!(SysCtlPeripheralReady(SYSCTL_PERIPH_SSI1)));

    //
    // For this example SSI1 is used with the following GPIO Pin Mapping
    // SSI1CLK   : PB5
    // SSI1FSS   : PB4
    // SSI1XDAT0 : PE4
    // SSI1XDAT1 : PE5
    // SSI1XDAT2 : PD4
    // SSI1XDAT3 : PD5
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    //
    // Configure the pin muxing for SSI1 functions on port B4, B5, E4, E5,
    // D4, D5
    //
    GPIOPinConfigure(GPIO_PB5_SSI1CLK);
    GPIOPinConfigure(GPIO_PB4_SSI1FSS);
    GPIOPinConfigure(GPIO_PE4_SSI1XDAT0);
    GPIOPinConfigure(GPIO_PE5_SSI1XDAT1);
    GPIOPinConfigure(GPIO_PD4_SSI1XDAT2);
    GPIOPinConfigure(GPIO_PD5_SSI1XDAT3);

    //
    // Configure the GPIO settings for the SSI pins.  This function also gives
    // control of these pins to the SSI hardware.  Consult the data sheet to
    // see which functions are allocated per pin.
    //
    GPIOPinTypeSSI(GPIO_PORTB_BASE, GPIO_PIN_5 | GPIO_PIN_4);
    GPIOPinTypeSSI(GPIO_PORTD_BASE, GPIO_PIN_5 | GPIO_PIN_4);
    GPIOPinTypeSSI(GPIO_PORTE_BASE, GPIO_PIN_5 | GPIO_PIN_4);

    HWREG(GPIO_PORTB_BASE+GPIO_O_DR8R) |= (GPIO_PIN_5 | GPIO_PIN_4);
    HWREG(GPIO_PORTD_BASE+GPIO_O_DR8R) |= (GPIO_PIN_5 | GPIO_PIN_4);
    HWREG(GPIO_PORTE_BASE+GPIO_O_DR8R) |= (GPIO_PIN_5 | GPIO_PIN_4);

    //
    // Configure and enable the SSI port for SPI master mode.  Use SSI1,
    // system clock supply, idle clock level low and active low clock in
    // freescale SPI mode, master mode, 60MHz SSI frequency, and 8-bit data.
    // For SPI mode, you can set the polarity of the SSI clock when the SSI
    // unit is idle.  You can also configure what clock edge you want to
    // capture data on.  Please reference the datasheet for more information on
    // the different SPI modes.
    //
    SSIConfigSetExpClk(SSI1_BASE, ui32SysClockFreq, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_MASTER, (ui32SysClockFreq/2), 8);

    //
    // Enable the SSI1 module.
    //
    SSIAdvModeSet(SSI1_BASE,SSI_ADV_MODE_WRITE);
    SSIAdvFrameHoldEnable(SSI1_BASE);
    HWREG(SSI1_BASE+SSI_O_CR1) |= SSI_CR1_HSCLKEN;
    SSIEnable(SSI1_BASE);

    //
    // Read any residual data from the SSI port.  This makes sure the receive
    // FIFOs are empty, so we don't read any unwanted junk.  This is done here
    // because the SPI SSI mode is full-duplex, which allows you to send and
    // receive at the same time.  The SSIDataGetNonBlocking function returns
    // "true" when data was returned, and "false" when no data was returned.
    // The "non-blocking" function checks if there is any data in the receive
    // FIFO and does not "hang" if there isn't.
    //
    while(SSIDataGetNonBlocking(SSI1_BASE, &pui32Dummy[0]))
    {
    }

    //
    // Initialize the Transmit Buffer
    //
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	pui32DataTx[ui32Index] = rand()%256;
    	pui32DataRx[ui32Index] = 0x0;
    }

    //
    // First Read the DEVICE ID
    //
    ui32DeviceID = SSILibSendReadIDAdvMode(SSI1_BASE);
    if(ui32DeviceID != 0xC219)
    {
       	UARTprintf("ERROR: No External Flash... Read Back %x\n",ui32DeviceID);
    	return (0);
    }

   	UARTprintf("QSSI : External Flash Detected with Device ID 0x%x\n",ui32DeviceID);

   	//
   	// Erase the Sector before Program Operation...
   	//
   	UARTprintf("ADV  : Starting Erase Operations...\n");
   	SSILibSendEraseCommand(SSI1_BASE,0x0,INS_SECTOR_ERASE_4KB);
   	UARTprintf("ADV  : Erase Completed...\n");

    //
    // Ensure Previous Operation is complete
    //
    SSILibDeviceBusyCheck(SSI1_BASE);

   	//
   	// Read NUM_SSI_DATA words from the External Flash
   	//
   	UARTprintf("ADV  : Starting Read Operations for Erase ...\n");
   	if(!(SSILibSendReadDataAdvBiQuad(SSI1_BASE,0x0,INS_ADV_READ_DATA)))
   	{
   	   	UARTprintf("ERROR: Wrong Command for Read Programming...\n");
   	   	return (0);
   	}

   	for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
        SSIDataPut(SSI1_BASE,DUMMY_BYTE);
        SSIDataGet(SSI1_BASE,&pui32DataRx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,DUMMY_BYTE);
    SSIDataGet(SSI1_BASE,&pui32DataRx[NUM_SSI_DATA-1]);
   	UARTprintf("ADV  : Read Completed for Erase Serial Flash...\n");

   	//
   	// Check Data and Display if Mismatch for Erase Operation and
   	// exit the main program
   	//
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	if(pui32DataRx[ui32Index] != 0xFF)
    	{
        	UARTprintf("ERROR: Erase Failure @ %d with value as %x\n",ui32Index, pui32DataRx[ui32Index]);
        	bErrorFlag = true;
    	}
    }

    if(bErrorFlag)
    {
    	return (0);
    }

   	//
   	// Write NUM_SSI_DATA words to the External Flash in Advanced Mode.
   	//
   	UARTprintf("=====================================\n");
   	UARTprintf("ADV  : Starting Program Operations...\n");
   	if(!(SSILibSendPageProgram(SSI1_BASE,0x0,INS_ADV_PAGE_PROGRAM)))
   	{
   	   	UARTprintf("ERROR: Wrong Command for Page Programming...\n");
   	   	return (0);
   	}

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPut(SSI1_BASE,pui32DataTx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,pui32DataTx[NUM_SSI_DATA-1]);

    //
    // Ensure Previous Operation is complete
    //
    SSILibDeviceBusyCheck(SSI1_BASE);

    //
    // Read data in Advanced Mode
    //
    SSILibSendReadDataAdvBiQuad(SSI1_BASE,0x0,INS_ADV_READ_DATA);

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPutNonBlocking(SSI1_BASE,DUMMY_BYTE);
        SSIDataGet(SSI1_BASE,&pui32DataRx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,DUMMY_BYTE);
    SSIDataGet(SSI1_BASE,&pui32DataRx[NUM_SSI_DATA-1]);

   	//
   	// Check Data and Display if Mismatch
   	//
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	if(pui32DataTx[ui32Index] != pui32DataRx[ui32Index])
    	{
        	UARTprintf("ERROR: Program: %x Read: %x\n",pui32DataTx[ui32Index],pui32DataRx[ui32Index]);
           	bErrorFlag = true;
    	}
    }

    if(bErrorFlag)
    {
    	return (0);
    }

   	UARTprintf("ADV  : Read Completed...\n");

   	//
   	// Write NUM_SSI_DATA words to the External Flash in Advanced Mode.
   	//
   	UARTprintf("=====================================\n");
   	UARTprintf("QUAD : Starting Program Operations...\n");
   	if(!(SSILibSendPageProgram(SSI1_BASE,0x100,INS_QUAD_PAGE_PROGRAM)))
   	{
   	   	UARTprintf("ERROR: Wrong Command for Page Programming...\n");
   	   	return (0);
   	}

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPut(SSI1_BASE,pui32DataTx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,pui32DataTx[NUM_SSI_DATA-1]);

    //
    // Ensure Previous Operation is complete
    //
    SSILibDeviceBusyCheck(SSI1_BASE);

    //
    // Read data in Bi Mode
    //
   	SSILibSendReadDataAdvBiQuad(SSI1_BASE,0x100,INS_BI_READ_DATA);

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPutNonBlocking(SSI1_BASE,DUMMY_BYTE);
        SSIDataGet(SSI1_BASE,&pui32DataRx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,DUMMY_BYTE);
    SSIDataGet(SSI1_BASE,&pui32DataRx[NUM_SSI_DATA-1]);

   	//
   	// Check Data and Display if Mismatch
   	//
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	if(pui32DataTx[ui32Index] != pui32DataRx[ui32Index])
    	{
        	UARTprintf("ERROR: Program: %x Read: %x\n",pui32DataTx[ui32Index],pui32DataRx[ui32Index]);
           	bErrorFlag = true;
    	}
    }

    if(bErrorFlag)
    {
    	return (0);
    }

   	UARTprintf("BI   : Read Completed...\n");

   	//
   	// Write NUM_SSI_DATA words to the External Flash in Advanced Mode.
   	//
   	UARTprintf("=====================================\n");
   	UARTprintf("ADV  : Starting Program Operations...\n");
   	if(!(SSILibSendPageProgram(SSI1_BASE,0x200,INS_ADV_PAGE_PROGRAM)))
   	{
   	   	UARTprintf("ERROR: Wrong Command for Page Programming...\n");
   	   	return (0);
   	}

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPut(SSI1_BASE,pui32DataTx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,pui32DataTx[NUM_SSI_DATA-1]);

    //
    // Ensure Previous Operation is complete
    //
    SSILibDeviceBusyCheck(SSI1_BASE);

    //
    // Read data in Quad Mode
    //
   	SSILibSendReadDataAdvBiQuad(SSI1_BASE,0x200,INS_QUAD_READ_DATA);

    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
    	SSIDataPutNonBlocking(SSI1_BASE,DUMMY_BYTE);
        SSIDataGet(SSI1_BASE,&pui32DataRx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,DUMMY_BYTE);
    SSIDataGet(SSI1_BASE,&pui32DataRx[NUM_SSI_DATA-1]);

   	//
   	// Check Data and Display if Mismatch
   	//
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	if(pui32DataTx[ui32Index] != pui32DataRx[ui32Index])
    	{
        	UARTprintf("ERROR: Program: %x Read: %x\n",pui32DataTx[ui32Index],pui32DataRx[ui32Index]);
           	bErrorFlag = true;
    	}
    }

    if(bErrorFlag)
    {
    	return (0);
    }

   	UARTprintf("QUAD : Read Completed...\n");

   	//
   	// Erase the Sector before Program Operation...
   	//
   	UARTprintf("=====================================\n");
   	UARTprintf("ADV  : Starting Erase Operations...\n");
   	SSILibSendEraseCommand(SSI1_BASE,0x0,INS_SECTOR_ERASE_4KB);
   	UARTprintf("ADV  : Erase Completed...\n");

   	//
   	// Read NUM_SSI_DATA words from the External Flash
   	//
   	UARTprintf("ADV  : Starting Read Operations for Erase ...\n");
   	SSILibSendReadDataAdvBiQuad(SSI1_BASE,0x0,INS_ADV_READ_DATA);
    for(ui32Index=0;ui32Index<NUM_SSI_DATA-1;ui32Index++)
    {
        SSIDataPut(SSI1_BASE,DUMMY_BYTE);
        SSIDataGet(SSI1_BASE,&pui32DataRx[ui32Index]);
    }
    SSIAdvDataPutFrameEnd(SSI1_BASE,DUMMY_BYTE);
    SSIDataGet(SSI1_BASE,&pui32DataRx[NUM_SSI_DATA-1]);
   	UARTprintf("ADV  : Read Completed for Erase Serial Flash...\n");

   	//
   	// Check Data and Display if Mismatch for Erase Operation
   	//
    for(ui32Index=0;ui32Index<NUM_SSI_DATA;ui32Index++)
    {
    	if(pui32DataRx[ui32Index] != 0xFF)
    	{
        	UARTprintf("ERROR: Erase Failure @ %d with value as %x\n",ui32Index, pui32DataRx[ui32Index]);
           	bErrorFlag = true;
    	}
    }

    if(bErrorFlag)
    {
    	return (0);
    }

    //
    // If code execution reaches here then all is success
    //
   	UARTprintf("QSSI : Serial Flash Erased...\n");

   	//
   	// Halt the Program in a while loop
   	//
   	while(1);

}
