/**************************************************************************************************
  Filename:       _hal_capsense.c
  Revised:        $Date: 2010-05-25 17:49:03 -0700 (Tue, 25 May 2010) $
  Revision:       $Revision: 22624 $

  Description:    This file contains the interface to the HAL KEY Service.


  Copyright 2006-2009 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, 
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE, 
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com. 
**************************************************************************************************/
/*********************************************************************
 NOTE: If polling is used, the hal_driver task schedules the KeyRead()
       to occur every 100ms.  This should be long enough to naturally
       debounce the keys.  The KeyRead() function remembers the key
       state of the previous poll and will only return a non-zero
       value if the key state changes.

 NOTE: If interrupts are used, the KeyRead() function is scheduled
       25ms after the interrupt occurs by the ISR.  This delay is used
       for key debouncing.  The ISR disables any further Key interrupt
       until KeyRead() is executed.  KeyRead() will re-enable Key
       interrupts after executing.  Unlike polling, when interrupts
       are enabled, the previous key state is not remembered.  This
       means that KeyRead() will return the current state of the keys
       (not a change in state of the keys).

*********************************************************************/

/**************************************************************************************************
 *                                            INCLUDES
 **************************************************************************************************/
#include "hal_mcu.h"
#include "hal_defs.h"
#include "hal_types.h"
#include "hal_drivers.h"
#include "hal_sleep.h"
#include "hal_adc.h"
#include "hal_capsense.h"
#include "osal.h"


// Compiler check
#if ((defined RF_STATS_RX) && (RF_STATS_RX == TRUE)) || ((defined RF_STATS_TX) && (RF_STATS_TX == TRUE))
#include "hal_assert.h"
#if (!defined UART_STATS) || (UART_STATS == FALSE)
#warning RF_STATS is only available if UART_STATS is defined and TRUE
#endif
#endif

/**************************************************************************************************
 *                                              MACROS
 **************************************************************************************************/

#define INIT_TIMER1_CHANNEL0() (T1CCTL0 = (T1CCTL0 & ~(HAL_CAPSENSE_T1CCTL0_CAP | HAL_CAPSENSE_T1CCTL0_MODE)) )
#define INIT_TIMER1_CHANNEL1() (T1CCTL1 = (T1CCTL1 & ~(HAL_CAPSENSE_T1CCTL1_CAP | HAL_CAPSENSE_T1CCTL1_MODE)) ) 
#define INIT_TIMER1_CHANNEL2() (T1CCTL2 = (T1CCTL2 & ~(HAL_CAPSENSE_T1CCTL2_CAP | HAL_CAPSENSE_T1CCTL2_MODE)) )
#define INIT_TIMER1_CHANNEL3() (T1CCTL3 = (T1CCTL3 & ~(HAL_CAPSENSE_T1CCTL3_CAP | HAL_CAPSENSE_T1CCTL3_MODE)) )
#define INIT_TIMER1_CHANNEL4() (T1CCTL4 = (T1CCTL4 & ~(HAL_CAPSENSE_T1CCTL4_CAP | HAL_CAPSENSE_T1CCTL4_MODE)) )

#define CAP_TIMER1_CHANNEL0() (T1CCTL0 = (T1CCTL0 & ~(HAL_CAPSENSE_T1CCTL0_CAP)) | HAL_CAPSENSE_T1C0_RISE_EDGE)
#define CAP_TIMER1_CHANNEL1() (T1CCTL1 = (T1CCTL1 & ~(HAL_CAPSENSE_T1CCTL1_CAP)) | HAL_CAPSENSE_T1C1_FALL_EDGE) 
#define CAP_TIMER1_CHANNEL2() (T1CCTL2 = (T1CCTL2 & ~(HAL_CAPSENSE_T1CCTL2_CAP)) | HAL_CAPSENSE_T1C2_FALL_EDGE)
#define CAP_TIMER1_CHANNEL3() (T1CCTL3 = (T1CCTL3 & ~(HAL_CAPSENSE_T1CCTL3_CAP)) | HAL_CAPSENSE_T1C3_FALL_EDGE)
#define CAP_TIMER1_CHANNEL4() (T1CCTL4 = (T1CCTL4 & ~(HAL_CAPSENSE_T1CCTL4_CAP)) | HAL_CAPSENSE_T1C4_FALL_EDGE)

#define NOCAP_TIMER1_CHANNEL0() (T1CCTL0 = (T1CCTL0 & ~(HAL_CAPSENSE_T1CCTL0_CAP)) )
#define NOCAP_TIMER1_CHANNEL1() (T1CCTL1 = (T1CCTL1 & ~(HAL_CAPSENSE_T1CCTL1_CAP)) ) 
#define NOCAP_TIMER1_CHANNEL2() (T1CCTL2 = (T1CCTL2 & ~(HAL_CAPSENSE_T1CCTL2_CAP)) )
#define NOCAP_TIMER1_CHANNEL3() (T1CCTL3 = (T1CCTL3 & ~(HAL_CAPSENSE_T1CCTL3_CAP)) )
#define NOCAP_TIMER1_CHANNEL4() (T1CCTL4 = (T1CCTL4 & ~(HAL_CAPSENSE_T1CCTL4_CAP)) )

#define INIT_TIMERS()     \
{                         \
  INIT_TIMER1_CHANNEL0(); \
  INIT_TIMER1_CHANNEL1(); \
  INIT_TIMER1_CHANNEL2(); \
  INIT_TIMER1_CHANNEL3(); \
  INIT_TIMER1_CHANNEL4(); \
}

#define START_TIMER_MODE_FREERUN()                                                                                     \
{                                                                                                                      \
  T1CNTL = 0x00; /*Reset timer value low byte*/                        \
  T1CNTH = 0x00; /*Reset timer value high byte*/                       \
  T1CTL = (T1CTL & ~(HAL_CAPSENSE_T1CTL_MODE | HAL_CAPSENSE_T1CTL_DIV)) | HAL_CAPSENSE_T1CTL_MODE_FREERUN; /*Start Timer 1. Free running mode*/ \
}
// | HAL_CAPSENSE_T1CTL_DIV_8

#define STOP_TIMER()                                            \
{                                                               \
  T1CTL = (T1CTL & ~(HAL_CAPSENSE_T1CTL_MODE)); /*Stop timer*/   \
}

#define DISABLE_INTERRUPTS()                                            \
{                                                                       \
  PERCFG  |= HAL_CAPSENSE_PERCFG_T1CFG; /*Setup Alternative 2 location of Timer1 I/O*/    \
  T1CCTL0 &= ~HAL_CAPSENSE_T1CCTL0_IM; /*Disable interrupt on Timer 1 - channel 0.*/ \
  T1CCTL1 &= ~HAL_CAPSENSE_T1CCTL1_IM; /*Disable interrupt on Timer 1 - channel 1.*/ \
  T1CCTL2 &= ~HAL_CAPSENSE_T1CCTL2_IM; /*Disable interrupt on Timer 1 - channel 2.*/ \
  T1CCTL3 &= ~HAL_CAPSENSE_T1CCTL3_IM; /*Disable interrupt on Timer 1 - channel 3.*/ \
  T1CCTL4 &= ~HAL_CAPSENSE_T1CCTL4_IM; /*Disable interrupt on Timer 1 - channel 4.*/ \
  /*T3CCTL0 &= ~HAL_CAPSENSE_T3CCTL0_IM; Disable interrupt on Timer 3 - channel 0.*/ \
  TIMIF &= ~BIT6;        /*Disable overflow interrupt OVFIM*/           \
}
  
#define SETUP_PERIPHERAL_IO() (P2SEL = (P2SEL & ~HAL_CAPSENSE_P2SEL_PRI1P1)| HAL_CAPSENSE_P2SEL_PRI0P1)

/*  
 *  Following defines are only valid for TI's STB FrontPanel
 *
 */
#if defined (HAL_BOARD_CC2531STBFP)

#define HAL_CAPSENSE_CHARGE(v)             CAP_CHARGE(v)
#define HAL_CAPSENSE_DISCHARGE(v)          CAP_DISCHARGE(v)

#define HAL_CAPSENSE_LOW                   0
#define HAL_CAPSENSE_HIGH                  1

#define CAP_CHARGE(MODE)      \
{                             \
  if(MODE & HAL_CAPSENSE_MODE_UP){         \
    /*DISCHARGE_P1_2();*/         \
    NOCAP_TIMER1_CHANNEL0();  \
    CHARGE_P1_1();            \
    NOCAP_TIMER1_CHANNEL2();  \
    CHARGE_P0_6();            \
    NOCAP_TIMER1_CHANNEL3();  \
  }                           \
  else if(MODE & HAL_CAPSENSE_MODE_DOWN){  \
    DISCHARGE_P1_2();         \
    CHARGE_P1_0();            \
    NOCAP_TIMER1_CHANNEL1();  \
    CHARGE_P0_7();            \
    NOCAP_TIMER1_CHANNEL4();  \
  }                           \
}                             

#define CAP_DISCHARGE(MODE)   \
{                             \
  if(MODE & HAL_CAPSENSE_MODE_UP){         \
    /*CHARGE_P1_2();*/            \
    DISCHARGE_P1_1();         \
    CAP_TIMER1_CHANNEL1();    \
    DISCHARGE_P0_6();         \
    CAP_TIMER1_CHANNEL4();    \
  }                           \
  else if(MODE & HAL_CAPSENSE_MODE_DOWN){  \
    CHARGE_P1_2();            \
    CAP_TIMER1_CHANNEL0();    \
    DISCHARGE_P1_0();         \
    CAP_TIMER1_CHANNEL2();    \
    DISCHARGE_P0_7();         \
    CAP_TIMER1_CHANNEL3();    \
  }                           \
}

#define CHARGE_P1_2()                                               \
{/*Standby button*/                                                 \
  P1SEL |= BIT2; /*P1_2 set for peripheral function, i.e. HZ*/     \
  P1DIR &= ~BIT2;/*P1_2 set as input.*/                  \
  P1INP |= BIT2;/*P1_2 to input tristate.*/              \
}

#define DISCHARGE_P1_2()                                            \
{/*Standby button*/                                                 \
  P1SEL &= ~BIT2;/*P1_2 set for general purpose IO*/                \
  P1DIR |= BIT2;/*P1_2 set as output.*/                             \
  P1_2   = HAL_CAPSENSE_LOW;                                        \
}

#define CHARGE_P0_7()                                               \
{/*Channel down button*/                                            \
  P0SEL &= ~BIT7; /*P0_7 set for general purpose IO*/               \
  P0DIR |= BIT7;  /*P0_7 set as output.*/                           \
  P0_7   = HAL_CAPSENSE_HIGH;                                       \
    /* P0_7 will next be set to peripheral (with input tri-state)
     * control, i.e. discharge (in DISCHARGE_P0_7() ) 
     */                                                             \
}

#define DISCHARGE_P0_7()                                            \
{/*Channel down button. 
  * Remember to call CHARGE_P0_7 at least \tau before*/             \
  P0SEL &= ~BIT6; /*P0_6 set for general purpose IO*/               \
  P0DIR |= BIT6;  /*P0_6 set as output.*/                           \
  P0_6 = HAL_CAPSENSE_LOW;                                          \
  P0SEL |= BIT7; /*P0_7 set for peripheral function IO*/            \
  P0DIR &= ~BIT7;/*P0_7 set as input.*/                             \
  P0INP |= BIT7;/*P0_7 to input tristate.*/                         \
}

#define CHARGE_P0_6()                                               \
{/*Channel up button*/                                            \
  P0SEL &= ~BIT6; /*P0_6 set for general purpose IO*/               \
  P0DIR |= BIT6;  /*P0_6 set as output.*/                           \
  P0_6 = HAL_CAPSENSE_HIGH;                                         \
    /* P0_6 will next be set to peripheral (with input tri-state)
     * control, i.e. discharge (in DISCHARGE_P0_6() ) 
     */                                                             \
}

#define DISCHARGE_P0_6()                                            \
{/*Channel up button. 
  * Remember to call CHARGE_P0_6 at least \tau before*/             \
  P0SEL &= ~BIT7; /*P0_7 set for general purpose IO*/               \
  P0DIR |= BIT7;  /*P0_7 set as output.*/                           \
  P0_7 = HAL_CAPSENSE_LOW;                                          \
  P0SEL |= BIT6; /*P0_6 set for peripheral function IO*/            \
  P0DIR &= ~BIT6;/*P0_6 set as input.*/                             \
  P0INP |= BIT6;/*P0_6 to input tristate.*/                         \
}

#define CHARGE_P1_0()                                               \
{/*Volume down button*/                                            \
  P1SEL &= ~BIT0; /*P1_0 set for general purpose IO*/               \
  P1DIR |= BIT0;  /*P1_0 set as output.*/                           \
  P1_0 = HAL_CAPSENSE_HIGH;                                         \
    /* P1_0 will next be set to peripheral (with input tri-state)
     * control, i.e. discharge (in DISCHARGE_P1_0() ) 
     */                                                             \
}

#define DISCHARGE_P1_0()                                            \
{/*Volume down button. 
  * Remember to call CHARGE_P1_0 at least \tau before*/             \
  P1SEL &= ~BIT1; /*P1_1 set for general purpose IO*/               \
  P1DIR |= BIT1;  /*P1_1 set as output.*/                           \
  P1_1 = HAL_CAPSENSE_LOW;                                          \
  P1SEL |= BIT0; /*P1_0 set for peripheral function IO*/            \
  P1DIR &= ~BIT0;/*P1_0 set as input.*/                             \
  P1INP |= BIT0;/*P1_0 to input tristate.*/                         \
}

#define CHARGE_P1_1()                                               \
{/*Volume up button*/                                            \
  P1SEL &= ~BIT1; /*P1_1 set for general purpose IO*/               \
  P1DIR |= BIT1;  /*P1_1 set as output.*/                           \
  P1_1 = HAL_CAPSENSE_HIGH;                                         \
    /* P1_1 will next be set to peripheral (with input tri-state)
     * control, i.e. discharge (in DISCHARGE_P1_1() ) 
     */                                                             \
}

#define DISCHARGE_P1_1()                                            \
{/*Volume up button. 
  * Remember to call CHARGE_P1_1 at least \tau before*/             \
  P1SEL &= ~BIT0; /*P1_0 set for general purpose IO*/               \
  P1DIR |= BIT0;  /*P1_0 set as output.*/                           \
  P1_0 = HAL_CAPSENSE_LOW;                                          \
  P1SEL |= BIT1; /*P1_1 set for peripheral function IO*/            \
  P1DIR &= ~BIT1;/*P1_1 set as input.*/                             \
  P1INP |= BIT1;/*P1_1 to input tristate.*/                         \
}

#else
#error No board defined, CapSense will not work
#endif //defined (HAL_BOARD_CC2531STBFP)

/**************************************************************************************************
 *                                            CONSTANTS
 **************************************************************************************************/


/**************************************************************************************************
 *                                            TYPEDEFS
 **************************************************************************************************/

/**************************************************************************************************
 *                                        LOCAL VARIABLES
 **************************************************************************************************/
short btnBaseCap[HAL_CAPSENSE_NOF_BTNS];
#define HAL_CAPSENSE_ADV_TRACK_MAX_COUNT          200 // Reset every 8 seconds, if pollrate is 40ms
#define HAL_CAPSENSE_THRESHOLD_MIN                 20 // minimum Threshold, will be updated from statistical variance
#define HAL_CAPSENSE_THRESHOLD_MIN_DETECT          50 // minimum Threshold for detection, will be updated from statistical variance
#define HAL_CAPSENSE_THRESHOLD_MULTIPLIER           2 // threshold is equal to this number times the variance, if that product is > HAL_CAPSENSE_THRESHOLD_MIN
#define HAL_CAPSENSE_THRESHOLD_MULTIPLIER_DETECT    4 // threshold for detection is equal to this number times the variance, if that product is > HAL_CAPSENSE_THRESHOLD_MIN_DETECT

static uint16 btnBaseCapTrackCounter = 1;
short btnBaseCapMeanOld[HAL_CAPSENSE_NOF_BTNS] = {0};
short btnBaseCapMeanNew[HAL_CAPSENSE_NOF_BTNS] = {0};
short btnBaseCapSold[HAL_CAPSENSE_NOF_BTNS] = {0};
short btnBaseCapSnew[HAL_CAPSENSE_NOF_BTNS] = {0};
short btnBaseCapVar[HAL_CAPSENSE_NOF_BTNS] = {0};
short dischargeTime = 0;

/**************************************************************************************************
 *                                        GLOBAL VARIABLES
 **************************************************************************************************/
//static uint8 halCapSenseSavedButton;     /* used to store previous button pressed . N/A for now*/
static halCapSenseCBack_t pHalCapSenseProcessFunction;
static uint8 halCapSenseMode;            /* Mode used to tell CapSenseRead what mode is active  */
static uint16 halCapSenseBtnCfgd = 0;    /* Per bit to tell if corresponding button has configured 
                                          * a baseCap (typically set to zero on startup)        */

/**************************************************************************************************
 *                                        FUNCTIONS - Local
 **************************************************************************************************/
uint8 halCapSenseUpdateBase (uint8 idx);
uint16 halCapSenseRead ();
#ifdef UART_STATS
void printHex16(uint16 in, uint8 btnId);
void uart0Setup(void);
#endif //UART_STATS

/**************************************************************************************************
 *                                        FUNCTIONS - API
 **************************************************************************************************/
/**************************************************************************************************
 * @fn      HalCapSenseInit
 *
 * @brief   Initilize CapSense Service
 *
 * @param   none
 *
 * @return  None
 **************************************************************************************************/
void HalCapSenseInit( void )
{
#if (HAL_CAPSENSE == TRUE)
  
  // Disable interrupts
  DISABLE_INTERRUPTS();
  
  // Setup peripheral IO
  SETUP_PERIPHERAL_IO();
  
  // Initialize timers
  INIT_TIMERS();
      
#endif /* HAL_CAPSENSE */
}

/**************************************************************************************************
 * @fn      HalCapSenseConfig
 *
 * @brief   Configure the CapSense serivce
 *
 * @param   cback - pointer to the CallBack function
 *
 * @return  None
 **************************************************************************************************/
void HalCapSenseConfig(halCapSenseCBack_t cback)
{
#if (HAL_CAPSENSE == TRUE)
  halIntState_t intState;
      
  halCapSenseMode = (uint8)HAL_CAPSENSE_MODE_UP;
  
  /* Charge the next capacitators to be measured here to be sure they're charged in time for 
   * discharge time measurements. This will then be well in time of one \tau.
   */
  HAL_CAPSENSE_CHARGE(halCapSenseMode);
  
  /* Register the callback fucntion */
  pHalCapSenseProcessFunction = cback;
  
HAL_ENTER_CRITICAL_SECTION( intState );    // Hold off interrupts
  // Reset and start Timer 1. Free running mode
  START_TIMER_MODE_FREERUN();
  // Discharge the capacitators to be measured. Make sure they were charged first.
  HAL_CAPSENSE_DISCHARGE(halCapSenseMode);
HAL_EXIT_CRITICAL_SECTION( intState );   // Re-enable interrupts.

  osal_start_timerEx (Hal_TaskID, HAL_CAPSENSE_EVENT, HAL_CAPSENSE_POLLING_VALUE);    /* Kick off polling */

#ifdef UART_STATS
  uart0Setup();
#endif //UART_STATS  
  
#endif /* HAL_CAPSENSE */
}


/**************************************************************************************************
 * @fn      HalCapSensePoll
 *
 * @brief   Called by hal_driver to poll the CapSense buttons
 *
 * @param   None
 *
 * @return  None
 **************************************************************************************************/
void HalCapSensePoll (void)
{
#if (HAL_CAPSENSE == TRUE)

  uint16 button = HAL_CAPSENSE_NO_BTN;  
  halIntState_t intState;
  
  // Enable next mode, by invertion (only works if they're defined this way)
  halCapSenseMode = (0xFF)^halCapSenseMode;
  
HAL_ENTER_CRITICAL_SECTION( intState );    // Hold off interrupts
  // Stop timer to make sure no new captures are made
  STOP_TIMER();
  /* Charge the next capacitators to be measured here to be sure they're charged in time for 
   * discharge time measurements. This will then be well in time of one \tau.
   */
  HAL_CAPSENSE_CHARGE(halCapSenseMode);
HAL_EXIT_CRITICAL_SECTION( intState );   // Re-enable interrupts.
    
  button = halCapSenseRead();
  
      
HAL_ENTER_CRITICAL_SECTION( intState );    // Hold off interrupts
  // Reset and start Timer 1. Free running mode
  START_TIMER_MODE_FREERUN();
  // Discharge the capacitators to be measured. Make sure they were charged first.
  HAL_CAPSENSE_DISCHARGE(halCapSenseMode);
HAL_EXIT_CRITICAL_SECTION( intState );   // Re-enable interrupts.
    
  // Maybe the osal_timer should be reset already here... TBD

  /* Invoke Callback if new keys were depressed */
  if ((button != HAL_CAPSENSE_NO_BTN ) &&
      (pHalCapSenseProcessFunction))
  {
    (pHalCapSenseProcessFunction) (button);
  }
#endif /* HAL_CAPSENSE */

}

/**************************************************************************************************
 * @fn      halCapSenseReset
 *
 * @brief   Will allow the application to reset the statistics of the 
 *          HalCapSense module
 *
 *          The variables, btnBaseCapMeanOld[btnIdx], are used for counting the 
 *          number of settling sequences, and should be set to zero in combination 
 *          with setting halCapSenseBtnCfgd to zero.
 *
 **************************************************************************************************/
void halCapSenseReset (void)
{
  uint8 btnIdx;
  
  for ( btnIdx=0 ; btnIdx<HAL_CAPSENSE_NOF_BTNS ; btnIdx++) {
    btnBaseCapMeanOld[btnIdx] = 0;
    btnBaseCapVar[btnIdx] = 0;
  }
  halCapSenseBtnCfgd = 0;
}

/**************************************************************************************************
 * @fn      halCapSenseRead
 *
 * @brief   Read the current value of the buttons. Mode will switch between
 *          UP and DOWN from one call to the next.
 *          
 *          NOTE! It takes a few (3-4) iterations for the dischargeTime to settle. 
 *          Hence, to avoid artificial statistics skip a few rounds before updating 
 *          the base. A call to halCapSenseUpdateBase will inherently update statistics.
 *
 *
 * @param   mode - direction of charge, depending on PCB layout
 *
 * @return  button - button pressed, the one that will be reported.
 **************************************************************************************************/
uint16 halCapSenseRead ()
{

  uint16 button = HAL_CAPSENSE_NO_BTN;
  uint8 btnIdx, *pT1CC = (uint8 *)P_T1CC, btnTmp;

#if (HAL_CAPSENSE == TRUE)
  switch ((0xFF)^halCapSenseMode) {
  case (uint8)HAL_CAPSENSE_MODE_UP:
      //Update the base capacitance of the buttons in MODE UP.
      for ( btnIdx=0; btnIdx<HAL_CAPSENSE_NOF_BTNS ; btnIdx++) {
//        if (button == HAL_CAPSENSE_NO_BTN) { // No button set yet. Priority is set from btnIdx.
          if ( BV(btnIdx) & HAL_CAPSENSE_MODE_UP_MASK)  {// Choose those buttons who belong to UP
            dischargeTime = (*pT1CC); // Read low        
            pT1CC++;  // Go to high, same channel
            dischargeTime |= (*pT1CC)<<8; // Read high
            if (halCapSenseBtnCfgd & BV(btnIdx)) {
              btnTmp = halCapSenseUpdateBase(btnIdx); // compares with and updates from recently read dischargeTime
              if(btnTmp != HAL_CAPSENSE_NO_BTN_IDX)
                button |= BV(btnTmp);
            } else { // button does not have a baseCap configured              
              btnBaseCapMeanOld[btnIdx]++; // use this variable to count number of settling iterations
              if (btnBaseCapMeanOld[btnIdx] > HAL_CAPSENSE_NOF_SETTLING_ITERATIONS) {                
                btnBaseCapMeanOld[btnIdx] = dischargeTime; // set baseCap
                btnBaseCapMeanNew[btnIdx] = dischargeTime;
                halCapSenseBtnCfgd |= BV(btnIdx); // button's baseCap configured (it is ok after a few iterations)
              }
            }
          } else {
            pT1CC++;  // Go to high, same channel (necessary if not a button of this mode)
          }
          pT1CC++;  // Go to low, next channel
//        }
      }
    break;
  case (uint8)HAL_CAPSENSE_MODE_DOWN:
      //Update the base capacitance of the buttons in MODE DOWN.
      for ( btnIdx=0 ; btnIdx<HAL_CAPSENSE_NOF_BTNS ; btnIdx++) {
//        if (button == HAL_CAPSENSE_NO_BTN) { // No button set yet. Priority is set from btnIdx.
          if ( BV(btnIdx) & HAL_CAPSENSE_MODE_DOWN_MASK)  {// Choose those buttons who belong to DOWN
            dischargeTime = (*pT1CC); // Read low        
            pT1CC++;  // Go to high, same channel
            dischargeTime |= (*pT1CC)<<8; // Read high
            if (halCapSenseBtnCfgd & BV(btnIdx)) {
              btnTmp = halCapSenseUpdateBase(btnIdx); // compares with and updates from recently read dischargeTime
              if(btnTmp != HAL_CAPSENSE_NO_BTN_IDX) 
                button |= BV(btnTmp);
            } else { // button does not have a baseCap configured           
              btnBaseCapMeanOld[btnIdx]++; // use this variable to count number of settling iterations
              if (btnBaseCapMeanOld[btnIdx] > HAL_CAPSENSE_NOF_SETTLING_ITERATIONS) {                
                btnBaseCapMeanOld[btnIdx] = dischargeTime; // set baseCap
                btnBaseCapMeanNew[btnIdx] = dischargeTime;
                halCapSenseBtnCfgd |= BV(btnIdx); // button's baseCap configured (it is ok after a few iterations)
              }
            }
          } else {
            pT1CC++;  // Go to high, same channel (necessary if not a button of this mode)
          }
          pT1CC++;  // Go to low, next channel
//        }
      }
      /* Update track count once per iteration. Put here because HAL_CAPSENSE_MODE_DOWN 
      * is called after HAL_CAPSENSE_MODE_UP, hence all buttons has been processed 
      * at this stage.
      */
      if (halCapSenseBtnCfgd) {
        if ( btnBaseCapTrackCounter > HAL_CAPSENSE_ADV_TRACK_MAX_COUNT)
          btnBaseCapTrackCounter = 1;
        else
          btnBaseCapTrackCounter++;
      }
    break;
  default:
    break;
  }
#endif /* HAL_CAPSENSE */

  return button;
}

/**************************************************************************************************
 * @fn      halCapSenseUpdateBase
 *
 * @brief   Update base capacitance value. This is a time in us < 
 *
 * @param   idx               // index of the base to be updated
 *
 * @return  buttonPressed     // Equal to idx if this button was pressed, else equal to HAL_CAPSENSE_NO_BTN
 **************************************************************************************************/
uint8 halCapSenseUpdateBase (uint8 idx)
{
  short delta =0;
    
  delta = dischargeTime - btnBaseCapMeanOld[idx];
#ifdef UART_STATS
//  if (idx == 1)
    printHex16(dischargeTime, idx);
#endif //UART_STATS
  
  if (btnBaseCapTrackCounter == HAL_CAPSENSE_ADV_TRACK_MAX_COUNT) {
    // for every n'th track count, the measure of the variance is assumed good enough to be stored.
    btnBaseCapVar[idx] = (btnBaseCapVar[idx] + (btnBaseCapSnew[idx])/(btnBaseCapTrackCounter-1)) >> 1;
    btnBaseCapSold[idx] = 0;
#ifdef UART_STATS
//    printHex16(dischargeTime, idx);
#endif //UART_STATS
  }
  // Compare with HAL_CAPSENSE_THRESHOLD_MULTIPLIER times the variance, but minimum HAL_CAPSENSE_THRESHOLD_MIN
  if(delta > (short)MAX(HAL_CAPSENSE_THRESHOLD_MIN,(btnBaseCapVar[idx] * HAL_CAPSENSE_THRESHOLD_MULTIPLIER))) {
    // button press detected; report
    if(delta > (short)MAX(HAL_CAPSENSE_THRESHOLD_MIN_DETECT,(btnBaseCapVar[idx] * HAL_CAPSENSE_THRESHOLD_MULTIPLIER_DETECT))) {
#ifdef UART_STATS
//      printHex16(dischargeTime, idx);
#endif //UART_STATS
      return idx;
    } else
      return HAL_CAPSENSE_NO_BTN_IDX;
  } else if (delta < -(short)(MAX(HAL_CAPSENSE_THRESHOLD_MIN,(btnBaseCapVar[idx] * HAL_CAPSENSE_THRESHOLD_MULTIPLIER)))) {
#ifdef UART_STATS
//    printHex16(delta, idx);
#endif //UART_STATS
    // Track down, but only with boundary condition. This is to avoid sudden drops.
    delta = -(short)(MAX(HAL_CAPSENSE_THRESHOLD_MIN,(btnBaseCapVar[idx] * HAL_CAPSENSE_THRESHOLD_MULTIPLIER)));
  } 
  
  // If no button was detected, update statistics.
  if (btnBaseCapSold[idx] & 0x8000) {
    // Overflow detected.
#ifdef UART_STATS
//    printHex16(btnBaseCapSold[idx], idx);
#endif //UART_STATS
    return HAL_CAPSENSE_NO_BTN_IDX;
  }
  btnBaseCapMeanNew[idx] = btnBaseCapMeanOld[idx] + (delta)/(short)(btnBaseCapTrackCounter);
  /* if (|delta|/btnBaseCapTrackCounter < 1) no tracking will be done, hence add/subtract 1 
  * to/from mean. Will result in ping-pong effect, but better overall tracking.
  */
  if (!ABS((delta)/(short)(btnBaseCapTrackCounter))) {
    if (delta > 0)
      btnBaseCapMeanNew[idx]++;
    else 
      btnBaseCapMeanNew[idx]--;
  }
  btnBaseCapSnew[idx] = btnBaseCapSold[idx] + (delta)*((short)(dischargeTime) - (short)(btnBaseCapMeanNew[idx]));
  btnBaseCapMeanOld[idx] = btnBaseCapMeanNew[idx];
  btnBaseCapSold[idx] = btnBaseCapSnew[idx];
 
    
  return HAL_CAPSENSE_NO_BTN_IDX;
}

#ifdef UART_STATS
void uart0Setup(void)
{
  PERCFG &= ~(0x01);  // Alternative 1 for USART 0
  P2DIR &= ~(0xC0); // USART0 peripheral priority 1
  P0SEL |= 0x0C;  // Set P0_2 and P0_3 as peripheral pins
  U0UCR |= 0x80;   // Reset USART0
  U0CSR = 0x80;   // Set USART0 -> UART mode
//  U0GCR = 0x08;   // Generic settings, BAUD_E for 9600
//  U0BAUD = 59;    // BAUD_M for 9600
  U0GCR =   11;   // Generic settings, BAUD_E for 115000
  U0BAUD = 216;    // BAUD_M for 115000
}

void uartPutc(uint8 c, uint8 cond)
{
#if defined (RF_STATS_TX) && (RF_STATS_TX == TRUE)
  static uint8 *msgPtr, ptrCnt = 0;
  if (cond == RF_STATS_START_CONDITION)
  {
    msgPtr = osal_msg_allocate( RF_STATS_LEN + RF_STATS_HDR_LEN );
    ptrCnt = 0;
    msgPtr[ptrCnt++] = RF_STATS_LEN;
    msgPtr[ptrCnt++] = RF_STATS_CMD_BYTE_CAPSENSE;
    msgPtr[ptrCnt++] = c;
  }
  else if (cond == RF_STATS_END_CONDITION)
  {
    msgPtr[ptrCnt++] = c;
    // Assert length
    HAL_ASSERT(ptrCnt == (RF_STATS_LEN + RF_STATS_HDR_LEN));
    osal_msg_send( zidDongleTaskId, msgPtr );
  }
  else if (cond == RF_STATS_CONTINUE_CONDITION)
  {
    msgPtr[ptrCnt++] = c;
  }
#elif ((defined RF_STATS_RX) && (RF_STATS_RX == TRUE))
  // Do nothing
  (void) cond;
#else
  (void) cond;
  while((U0CSR & 0x01));  // Wait until ACTIVE bit is low
  U0DBUF = c;
#endif //defined (RF_STATS_TX) && (RF_STATS_TX == TRUE)  
}

void printHex16(uint16 in, uint8 btnId)
{
  uint16 temp;
  int i;
//  uartPutc(HI_UINT16(in));
//  uartPutc(LO_UINT16(in));
//  uartPutc('x');
  if (btnId == 1)    
    uartPutc((btnId + '0'), RF_STATS_START_CONDITION);
  else
    uartPutc((btnId + '0'), RF_STATS_CONTINUE_CONDITION);
  
  uartPutc(' ', RF_STATS_CONTINUE_CONDITION);
  for(i=3; i > -1; i--)
  {
    temp = in >> (4*i);
    temp &= 0x0f;
    if(temp < 10) uartPutc((temp + '0'), RF_STATS_CONTINUE_CONDITION);
    else uartPutc(((temp - 10) + 'A'), RF_STATS_CONTINUE_CONDITION);
  }
  uartPutc(' ', RF_STATS_CONTINUE_CONDITION);
  for(i=3; i > -1; i--)
  {
    temp = btnBaseCapMeanOld[btnId] >> (4*i);
    temp &= 0x0f;
    if(temp < 10) uartPutc((temp + '0'), RF_STATS_CONTINUE_CONDITION);
    else uartPutc(((temp - 10) + 'A'), RF_STATS_CONTINUE_CONDITION);
  }
  uartPutc(' ', RF_STATS_CONTINUE_CONDITION);
  for(i=3; i > -1; i--)
  {
    temp = btnBaseCapVar[btnId] >> (4*i);
    temp &= 0x0f;
    if(temp < 10) uartPutc((temp + '0'), RF_STATS_CONTINUE_CONDITION);
    else uartPutc(((temp - 10) + 'A'), RF_STATS_CONTINUE_CONDITION);
  }
  if (btnId == 3) {
    uartPutc('\r', RF_STATS_CONTINUE_CONDITION);
    uartPutc('\n', RF_STATS_END_CONDITION);
  } else {
    uartPutc(' ', RF_STATS_CONTINUE_CONDITION);
  }
}

#endif //UART_STATS

/**************************************************************************************************
**************************************************************************************************/
